/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: ExtendedMemoryDialog.cpp 12626 2018-10-30 14:55:29Z LuisContreras $:

  Description:
    Device Extended Memory Area Dialog

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2018-10-23  Replace rcX headers with Hilscher Definitions
    2011-12-19  Fixed converting and displaying 64 Bit pointer
    2011-12-16  Added access type to extended memory dialog
    2011-12-08  Fixed Parameter validation for extended memory dialog
    2011-12-08  Added extended memory dialog (MRAM)
    2006-07-04  initial version

**************************************************************************************/

#include "stdafx.h"
#include "cifXTest.h"
#include "ExtendedMemoryDialog.h"
#include "CifXDeviceBase.h"
#include "CifXTestDlg.h"
#include "Hil_DualPortMemory.h"

IMPLEMENT_DYNAMIC(CExtendedMemoryDialog, CDialog)

#define BYTES_PER_LINE  16

CExtendedMemoryDialog::CExtendedMemoryDialog(CWnd* pParent /*=NULL*/)
	: CBaseDialog(CExtendedMemoryDialog::IDD, pParent)
  , m_pcCurrentDevice(NULL)
  , m_ulRecvDataOffset(0)
  , m_ulRecvDataLen(0)
  , m_pbWriteData(NULL)
  , m_ulWriteDataLen(0)
  , m_ulWriteDataOffset(0)
{

}

CExtendedMemoryDialog::~CExtendedMemoryDialog()
{
  if(NULL != m_pbWriteData)
  {
    free(m_pbWriteData);
    m_pbWriteData = NULL;
  }
}

void CExtendedMemoryDialog::DoDataExchange(CDataExchange* pDX)
{
	CDialog::DoDataExchange(pDX);
}


BEGIN_MESSAGE_MAP(CExtendedMemoryDialog, CDialog)
  ON_BN_CLICKED(IDC_BTN_GETMEMPTR,  &CExtendedMemoryDialog::OnBnClickedBtnGetMemPtr)
  ON_BN_CLICKED(IDC_BTN_MEMREAD,    &CExtendedMemoryDialog::OnBnClickedBtnMemRead)
  ON_BN_CLICKED(IDC_BTN_MEMWRITE,   &CExtendedMemoryDialog::OnBnClickedBtnMemWrite)
  ON_EN_KILLFOCUS(IDC_MEMORY_WRITEOFFSET, &CExtendedMemoryDialog::OnEnKillfocusMemoryWriteoffset)
  ON_EN_KILLFOCUS(IDC_MEMORY_READOFFSET, &CExtendedMemoryDialog::OnEnKillfocusMemoryReadoffset)
  ON_EN_KILLFOCUS(IDC_MEMORY_READLENGTH, &CExtendedMemoryDialog::OnEnKillfocusMemoryReadlength)
END_MESSAGE_MAP()

void CExtendedMemoryDialog ::UpdateUI(void)
{
  if(NULL == m_pcCurrentDevice)
  {
    GetDlgItem(IDC_BTN_GETMEMPTR)->SetWindowText(_T("Map Memory Pointer"));
    GetDlgItem(IDC_EDT_MEMPTR)->SetWindowText(_T(""));
    GetDlgItem(IDC_EDT_MEMTYPE)->SetWindowText(_T(""));
    GetDlgItem(IDC_EDT_MEMACCESS)->SetWindowText(_T(""));
    GetDlgItem(IDC_EDT_MEMSIZE)->SetWindowText(_T("0"));

    GetDlgItem(IDC_BTN_GETMEMPTR)->EnableWindow(FALSE);

    GetDlgItem(IDC_BTN_MEMREAD)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_READLENGTH)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_READOFFSET)->EnableWindow(FALSE);

    GetDlgItem(IDC_BTN_MEMWRITE)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_WRITEOFFSET)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_WRITEDATA)->EnableWindow(FALSE);
  } else if(NULL == m_pcCurrentDevice->ExtendedMemoryPointer(NULL))
  {
    TCHAR     szTemp[16];
    uint32_t  ulSize, ulType;

    GetDlgItem(IDC_BTN_GETMEMPTR)->SetWindowText(_T("Map Memory Pointer"));
    GetDlgItem(IDC_EDT_MEMPTR)->SetWindowText(_T(""));

    m_pcCurrentDevice->ExtendedMemoryInfo(ulSize, ulType);
    GetDlgItem(IDC_EDT_MEMSIZE)->SetWindowText(_ltot( ulSize, szTemp, 10));
    GetDlgItem(IDC_EDT_MEMTYPE)->SetWindowText(_ltot( (ulType &HIL_SYSTEM_EXTMEM_TYPE_MSK), szTemp, 10));
    GetDlgItem(IDC_EDT_MEMACCESS)->SetWindowText(_ltot( (ulType & HIL_SYSTEM_EXTMEM_ACCESS_MSK) >> 6, szTemp, 10));

    GetDlgItem(IDC_BTN_GETMEMPTR)->EnableWindow(TRUE);

    GetDlgItem(IDC_BTN_MEMREAD)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_READLENGTH)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_READOFFSET)->EnableWindow(FALSE);

    GetDlgItem(IDC_BTN_MEMWRITE)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_WRITEOFFSET)->EnableWindow(FALSE);
    GetDlgItem(IDC_MEMORY_WRITEDATA)->EnableWindow(FALSE);
  } else if(NULL != m_pcCurrentDevice->ExtendedMemoryPointer(NULL))
  {
    TCHAR     szTemp[32];
    uint32_t  ulSize, ulType;

    GetDlgItem(IDC_BTN_GETMEMPTR)->SetWindowText(_T("Release Memory Pointer"));

    m_pcCurrentDevice->ExtendedMemoryInfo(ulSize, ulType);
    GetDlgItem(IDC_EDT_MEMTYPE)->SetWindowText(_ltot( (ulType &HIL_SYSTEM_EXTMEM_TYPE_MSK), szTemp, 10));
    GetDlgItem(IDC_EDT_MEMACCESS)->SetWindowText(_ltot( (ulType & HIL_SYSTEM_EXTMEM_ACCESS_MSK) >> 6, szTemp, 10));
    GetDlgItem(IDC_EDT_MEMSIZE)->SetWindowText(_ltot(ulSize, szTemp, 10));

    _sntprintf_s(szTemp, 32, 32, "0x%p", m_pcCurrentDevice->ExtendedMemoryPointer(NULL));
    GetDlgItem(IDC_EDT_MEMPTR)->SetWindowText(szTemp);

    GetDlgItem(IDC_BTN_GETMEMPTR)->EnableWindow(TRUE);

    GetDlgItem(IDC_BTN_MEMREAD)->EnableWindow(TRUE);
    GetDlgItem(IDC_MEMORY_READLENGTH)->EnableWindow(TRUE);
    GetDlgItem(IDC_MEMORY_READOFFSET)->EnableWindow(TRUE);

    GetDlgItem(IDC_BTN_MEMWRITE)->EnableWindow(TRUE);
    GetDlgItem(IDC_MEMORY_WRITEOFFSET)->EnableWindow(TRUE);
    GetDlgItem(IDC_MEMORY_WRITEDATA)->EnableWindow(TRUE);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Device has changed or Dialog has been activated
/////////////////////////////////////////////////////////////////////////////
void CExtendedMemoryDialog ::OnUpdateDevice(CCifXDeviceBase* pcDevice)
{
  if(pcDevice != m_pcCurrentDevice)
  {
    /* Device has changed, so release previous memory pointer and
       disable dialog elements that are not available */
    if(NULL != m_pcCurrentDevice)
    {
      m_pcCurrentDevice->ExtendedMemoryRelease();
    }
  }

  m_pcCurrentDevice = pcDevice;
  UpdateUI();
}

void CExtendedMemoryDialog::OnBnClickedBtnGetMemPtr()
{
  int32_t lRet;

  if(NULL != m_pcCurrentDevice->ExtendedMemoryPointer(NULL))
  {
    /* We need to unmap */
    lRet = m_pcCurrentDevice->ExtendedMemoryRelease();
  } else
  {
    /* We need to map pointer */
    lRet = m_pcCurrentDevice->ExtendedMemoryMap();
  }
  
  if(CIFX_NO_ERROR != lRet)
  {
    CString csError;
    csError.Format(_T("Error mapping/releasing memory (0x%08X) - \r\n"), lRet);
    csError.Append(CcifXTestDlg::s_pcDevice->GetErrorDescription(lRet));

    AfxMessageBox(csError, MB_OK);
  } else
  {
    UpdateUI();
  }
}

void CExtendedMemoryDialog::OnEnKillfocusMemoryReadoffset()
{
  CString csTemp;
  CWnd*   pcEditReadOffset = GetDlgItem(IDC_MEMORY_READOFFSET);

  pcEditReadOffset->GetWindowText(csTemp);
  ULONG ulOffset = _ttoi(csTemp);

  uint32_t ulSize;
  m_pcCurrentDevice->ExtendedMemoryPointer(&ulSize);

  if((ulOffset + m_ulRecvDataLen) > ulSize)
  {
    CString csError;
    csError.Format(_T("Requested offset out of memory area. Offset set to zero!"));
    AfxMessageBox(csError, MB_OK);

    /* Offset larger than memory area */
    ulOffset = 0;
  }

  m_ulRecvDataOffset = ulOffset;

  csTemp.Format(_T("%u"), ulOffset);
  pcEditReadOffset->SetWindowText(csTemp);
}

void CExtendedMemoryDialog::OnEnKillfocusMemoryReadlength()
{
  CString csTemp;
  CWnd*   pcEditReadLength = GetDlgItem(IDC_MEMORY_READLENGTH);

  pcEditReadLength->GetWindowText(csTemp);
  ULONG ulLength = _ttoi(csTemp);

  uint32_t ulSize;
  m_pcCurrentDevice->ExtendedMemoryPointer(&ulSize);

  if((m_ulRecvDataOffset + ulLength) > ulSize)
  {
    CString csError;
    csError.Format(_T("Requested length out of memory area. Length adjusted!"));
    AfxMessageBox(csError, MB_OK);

    /* Offset larger than memory area */
    ulLength = ulSize - m_ulRecvDataOffset;
  }

  m_ulRecvDataLen = ulLength;

  csTemp.Format(_T("%u"), ulLength);
  pcEditReadLength->SetWindowText(csTemp);
}

void CExtendedMemoryDialog::OnBnClickedBtnMemRead()
{
  BYTE*   pbExtMem = (BYTE*)m_pcCurrentDevice->ExtendedMemoryPointer(NULL);
  CString csTemp;

  for(ULONG ulIdx = 0; ulIdx < m_ulRecvDataLen; ++ulIdx)
  {
    if((ulIdx > 0) && ((ulIdx % BYTES_PER_LINE) == 0))
      csTemp.AppendFormat(_T("\r\n%02X "), pbExtMem[m_ulRecvDataOffset + ulIdx]);
    else
      csTemp.AppendFormat(_T("%02X "), pbExtMem[m_ulRecvDataOffset + ulIdx]);
  }

  GetDlgItem(IDC_MEMORY_READDATA)->SetWindowText(csTemp);
}

void CExtendedMemoryDialog::OnEnKillfocusMemoryWriteoffset()
{
  CString csTemp;
  CWnd*   pcEditWriteOffset = GetDlgItem(IDC_MEMORY_WRITEOFFSET);

  pcEditWriteOffset->GetWindowText(csTemp);
  ULONG ulOffset = _ttoi(csTemp);

  uint32_t ulSize;
  m_pcCurrentDevice->ExtendedMemoryPointer(&ulSize);

  if(ulOffset >= ulSize)
  {
    AfxMessageBox(_T("Offset exceeds memory size! Offset will be set to zero"), MB_OK);
    ulOffset = 0;
  }
  csTemp.Format(_T("%u"), ulOffset);
  m_ulWriteDataOffset = ulOffset;

  pcEditWriteOffset->SetWindowText(csTemp);
}

void CExtendedMemoryDialog::OnBnClickedBtnMemWrite()
{
  /* Validate Write data and throw warning if Offset + Length is out of memory area */
  CString csTemp;
  CWnd*   pcWriteDataWnd = GetDlgItem(IDC_MEMORY_WRITEDATA);

  pcWriteDataWnd->GetWindowText(csTemp);
  csTemp.Remove(_T(' '));
  csTemp.Remove(_T('\r'));
  csTemp.Remove(_T('\n'));
  csTemp.Remove(_T('\t'));

  if(csTemp.GetLength() % 2)
  {
    /* Missing a nibble, add a zero at the end and warn user */
    AfxMessageBox(  _T("The data you entered is missing one nibble!\r\n") \
                    _T("Length will be adjusted!"));

    csTemp = csTemp.Left(csTemp.GetLength() - 1);
  }

  ULONG ulNewWriteLen = csTemp.GetLength() / 2;
  
  if(m_ulWriteDataLen != ulNewWriteLen)
  {
    m_pbWriteData = (BYTE*)realloc(m_pbWriteData, ulNewWriteLen);    
  }

  m_ulWriteDataLen = ulNewWriteLen;
 
  /* Convert string to byte array */
  for(ULONG ulIdx = 0; ulIdx < m_ulWriteDataLen; ++ulIdx)
  {
    TCHAR* pszEnd = NULL;

    m_pbWriteData[ulIdx] = (BYTE)_tcstol(csTemp.Mid(ulIdx * 2, 2), &pszEnd, 16);
  }

  csTemp.Format(_T("%u"), m_ulWriteDataLen);
  GetDlgItem(IDC_MEMORY_WRITELENGTH)->SetWindowText(csTemp);

  /* Convert byte array to string for displaying */
  csTemp.Empty();

  for(ULONG ulIdx = 0; ulIdx < m_ulWriteDataLen; ++ulIdx)
  {
    if((ulIdx > 0) && ((ulIdx % BYTES_PER_LINE) == 0))
      csTemp.AppendFormat(_T("\r\n%02X "), m_pbWriteData[ulIdx]);
    else
      csTemp.AppendFormat(_T("%02X "), m_pbWriteData[ulIdx]);
  }

  pcWriteDataWnd->SetWindowText(csTemp);

  ULONG ulCopyLen = m_ulWriteDataLen;
  uint32_t  ulSize;

  BYTE* pbExtMem = (BYTE*)m_pcCurrentDevice->ExtendedMemoryPointer(&ulSize);
  if( (m_ulWriteDataOffset + m_ulWriteDataLen) > ulSize)
  {
    /* Data does not fit in memory window */
    ulCopyLen = ulSize - m_ulWriteDataOffset;

    CString csError;
    csError.Format( _T("Length of data exceeds extended memory!\r\n") \
                    _T("Only %u bytes will be copied!"),
                    ulCopyLen);
    AfxMessageBox(csError);
  }

  /* Write data to extended memory block */
  for(ULONG ulIdx = 0; ulIdx < ulCopyLen; ++ulIdx)
  {
    pbExtMem[m_ulWriteDataOffset + ulIdx] = m_pbWriteData[ulIdx];
  }
}

BOOL CExtendedMemoryDialog::OnInitDialog()
{
  CBaseDialog::OnInitDialog();

  GetDlgItem(IDC_MEMORY_READLENGTH)->SetWindowText(_T("0"));
  GetDlgItem(IDC_MEMORY_READOFFSET)->SetWindowText(_T("0"));
  GetDlgItem(IDC_MEMORY_WRITELENGTH)->SetWindowText(_T("0"));
  GetDlgItem(IDC_MEMORY_WRITEOFFSET)->SetWindowText(_T("0"));

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}
